/**
 * Salamandre Child Theme - Search Modal
 * Inspired by Twenty Twenty's toggle system
 * 
 * Gère l'ouverture/fermeture de la modal de recherche avec :
 * - Animation slide depuis le haut
 * - Overlay avec fond flouté
 * - Verrouillage du scroll
 * - Trap de focus pour l'accessibilité
 * - Fermeture par Échap ou clic extérieur
 * - Bouton clear pour effacer le champ
 */

(function() {
    'use strict';

    // Namespace
    var salamandreSearch = {
        scrolled: 0
    };

    /**
     * Initialise la modal de recherche
     */
    salamandreSearch.init = function() {
        var toggles = document.querySelectorAll('[data-toggle-target]');
        var modal = document.querySelector('.search-modal');

        // Vérifier que les éléments existent
        if (!modal || toggles.length === 0) {
            return;
        }

        // Configuration des boutons toggle
        this.setupToggles(toggles, modal);

        // Fermeture au clic extérieur
        this.closeOnOutsideClick(modal);

        // Fermeture à la touche Échap
        this.closeOnEscape(modal);

        // Gestion du focus (accessibilité)
        this.manageFocus(modal);

        // Gestion du bouton clear
        this.setupClearButton(modal);
    };

    /**
     * Configure les boutons d'ouverture/fermeture
     */
    salamandreSearch.setupToggles = function(toggles, modal) {
        toggles.forEach(function(toggle) {
            toggle.addEventListener('click', function(e) {
                e.preventDefault();
                salamandreSearch.toggleModal(modal, toggle);
            });
        });
    };

    /**
     * Bascule l'état de la modal (ouvert/fermé)
     */
    salamandreSearch.toggleModal = function(modal, toggle) {
        var isActive = modal.classList.contains('active');

        if (isActive) {
            this.closeModal(modal, toggle);
        } else {
            this.openModal(modal, toggle);
        }
    };

    /**
     * Ouvre la modal
     */
    salamandreSearch.openModal = function(modal, toggle) {
        // Affiche la modal (display: block)
        modal.classList.add('show-modal');

        // Délai pour permettre la transition CSS
        setTimeout(function() {
            modal.classList.add('active');
            toggle.setAttribute('aria-expanded', 'true');
            modal.setAttribute('aria-hidden', 'false');
            
            document.body.classList.add('showing-modal');

            // Focus automatique sur le champ de recherche
            var searchField = modal.querySelector('.search-field');
            if (searchField) {
                setTimeout(function() {
                    searchField.focus();
                }, 300);
            }

            // Mettre à jour la visibilité du bouton clear
            salamandreSearch.updateClearButton(modal);
        }, 10);
    };

    /**
     * Ferme la modal
     */
    salamandreSearch.closeModal = function(modal, toggle) {
        modal.classList.remove('active');
        
        if (toggle) {
            toggle.setAttribute('aria-expanded', 'false');
        }
        
        modal.setAttribute('aria-hidden', 'true');
        document.body.classList.remove('showing-modal');

        // Masque la modal après l'animation
        setTimeout(function() {
            modal.classList.remove('show-modal');
        }, 300);

        // Retour du focus sur le bouton
        if (toggle) {
            toggle.focus();
        }
    };

    /**
     * Ferme la modal au clic sur l'overlay
     */
    salamandreSearch.closeOnOutsideClick = function(modal) {
        modal.addEventListener('click', function(e) {
            // Ne ferme que si on clique sur l'overlay, pas sur le contenu
            if (e.target === modal) {
                var toggle = document.querySelector('[data-toggle-target=".search-modal"]');
                salamandreSearch.closeModal(modal, toggle);
            }
        });
    };

    /**
     * Ferme la modal avec la touche Échap
     */
    salamandreSearch.closeOnEscape = function(modal) {
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape' && modal.classList.contains('active')) {
                var toggle = document.querySelector('[data-toggle-target=".search-modal"]');
                salamandreSearch.closeModal(modal, toggle);
            }
        });
    };

    /**
     * Gère le trap de focus (accessibilité)
     * Empêche le focus de sortir de la modal quand elle est ouverte
     */
    salamandreSearch.manageFocus = function(modal) {
        modal.addEventListener('keydown', function(e) {
            if (!modal.classList.contains('active')) {
                return;
            }

            // Récupère tous les éléments focusables
            var focusableElements = modal.querySelectorAll(
                'button:not([disabled]), [href], input, select, textarea, [tabindex]:not([tabindex="-1"])'
            );

            if (focusableElements.length === 0) {
                return;
            }

            var firstElement = focusableElements[0];
            var lastElement = focusableElements[focusableElements.length - 1];

            // Gère la touche Tab
            if (e.key === 'Tab') {
                if (e.shiftKey) {
                    // Shift + Tab : du premier au dernier
                    if (document.activeElement === firstElement) {
                        e.preventDefault();
                        lastElement.focus();
                    }
                } else {
                    // Tab : du dernier au premier
                    if (document.activeElement === lastElement) {
                        e.preventDefault();
                        firstElement.focus();
                    }
                }
            }
        });
    };

    /**
     * Configure le bouton clear
     */
    salamandreSearch.setupClearButton = function(modal) {
        var searchField = modal.querySelector('.search-field');
        var clearButton = modal.querySelector('.search-clear');

        if (!searchField || !clearButton) {
            return;
        }

        // Afficher/masquer le bouton selon le contenu du champ
        searchField.addEventListener('input', function() {
            salamandreSearch.updateClearButton(modal);
        });

        // Action du bouton clear
        clearButton.addEventListener('click', function(e) {
            e.preventDefault();
            searchField.value = '';
            searchField.focus();
            salamandreSearch.updateClearButton(modal);
        });
    };

    /**
     * Met à jour la visibilité du bouton clear
     */
    salamandreSearch.updateClearButton = function(modal) {
        var searchField = modal.querySelector('.search-field');
        var clearButton = modal.querySelector('.search-clear');

        if (!searchField || !clearButton) {
            return;
        }

        if (searchField.value.length > 0) {
            clearButton.classList.add('visible');
        } else {
            clearButton.classList.remove('visible');
        }
    };

    /**
     * Utilitaire : vérifie si le DOM est prêt
     */
    function domReady(fn) {
        if (document.readyState === 'interactive' || document.readyState === 'complete') {
            fn();
        } else {
            document.addEventListener('DOMContentLoaded', fn);
        }
    }

    // Initialisation au chargement du DOM
    domReady(function() {
        salamandreSearch.init();
    });

})();